(() => {
    const parseNum = (v) => {
        const n = parseFloat(String(v || '').replace(',', '.'));
        return Number.isFinite(n) ? n : 0;
    };

    // Copy helpers (e.g. customer suggestions link)
    document.querySelectorAll('[data-copy-target]').forEach((btn) => {
        btn.addEventListener('click', async () => {
            const sel = btn.getAttribute('data-copy-target');
            if (!sel) return;
            const el = document.querySelector(sel);
            const text = el && 'value' in el ? String(el.value || '') : String(el?.textContent || '');
            if (!text) return;

            const original = btn.textContent;
            try {
                await navigator.clipboard.writeText(text);
            } catch {
                try {
                    // Fallback
                    if (el && 'select' in el) {
                        el.select();
                        document.execCommand('copy');
                    }
                } catch {
                    // ignore
                }
            }

            btn.textContent = 'Nokopēts';
            window.setTimeout(() => {
                btn.textContent = original;
            }, 1200);
        });
    });

    // Labor amount preview
    const laborForm = document.querySelector('form[action$="/labor"]');
    const laborPreview = document.getElementById('laborAmountPreview');
    if (laborForm && laborPreview) {
        const unitTypeSelect = laborForm.querySelector('select[name="unit_type"]');
        const minutesInput = laborForm.querySelector('input[name="minutes"]');
        const qtyInput = laborForm.querySelector('input[name="qty"]');
        const rateInput = laborForm.querySelector('input[name="rate"]');
        const discountInput = laborForm.querySelector('input[name="discount_percent"]');
        const userSelect = laborForm.querySelector('select[name="user_id"]');
        const unitFields = laborForm.querySelectorAll('[data-unit-field]');

        const unitType = () => (unitTypeSelect?.value || 'hour').toLowerCase();

        const autoRateFromSelection = () => {
            if (unitType() !== 'hour') return 0;
            const fallback = parseNum(rateInput?.getAttribute('data-fallback-rate'));
            const personal = parseNum(rateInput?.getAttribute('data-default-rate'));

            // Manager: selected mechanic hourly rate (if available)
            if (userSelect instanceof HTMLSelectElement) {
                const opt = userSelect.selectedOptions?.[0];
                const salaryType = (opt?.getAttribute('data-salary-type') || '').toLowerCase();
                const hr = parseNum(opt?.getAttribute('data-hourly-rate'));
                if (salaryType === 'hourly' && hr > 0) return hr;
            }

            // Mechanic: own hourly rate injected by the server
            if (personal > 0) return personal;

            // Global default (settings)
            if (fallback > 0) return fallback;

            return 0;
        };

        const syncRatePlaceholder = () => {
            if (!(rateInput instanceof HTMLInputElement)) return;
            if (String(rateInput.value || '').trim() !== '') return;
            const r = autoRateFromSelection();
            const suffix = unitType() === 'piece' ? 'EUR/gab.' : 'EUR/h';
            rateInput.setAttribute('placeholder', r > 0 ? `${suffix} (auto: ${r.toFixed(2)})` : suffix);
        };

        const syncUnitFields = () => {
            const type = unitType();
            unitFields.forEach((wrap) => {
                const match = wrap.getAttribute('data-unit-field') === type;
                wrap.classList.toggle('d-none', !match);
            });
            if (minutesInput) minutesInput.required = type === 'hour';
            if (qtyInput) qtyInput.required = type === 'piece';
        };

        const syncLabor = () => {
            const minutes = parseInt(String(minutesInput?.value || '0'), 10) || 0;
            const qty = parseNum(qtyInput?.value);
            const typedRate = parseNum(rateInput?.value);
            const rate = typedRate > 0 ? typedRate : autoRateFromSelection();
            const gross = unitType() === 'piece'
                ? Math.round((qty * rate) * 100) / 100
                : Math.round(((minutes / 60) * rate) * 100) / 100;
            const disc = Math.min(100, Math.max(0, parseNum(discountInput?.value)));
            const amount = Math.round((gross * (1 - disc / 100)) * 100) / 100;
            const auto = typedRate <= 0 && rate > 0;
            const hasQty = unitType() === 'piece' ? qty > 0 : minutes > 0;
            laborPreview.textContent = hasQty && rate > 0
                ? `Summa: ${amount.toFixed(2)} EUR${disc > 0 ? ` (-${disc.toFixed(2)}%)` : ''}${auto ? ' (auto)' : ''}`
                : 'Summa: —';
        };

        unitTypeSelect?.addEventListener('change', () => {
            syncUnitFields();
            syncRatePlaceholder();
            syncLabor();
        });
        minutesInput?.addEventListener('input', syncLabor);
        qtyInput?.addEventListener('input', syncLabor);
        discountInput?.addEventListener('input', syncLabor);
        rateInput?.addEventListener('input', () => {
            syncRatePlaceholder();
            syncLabor();
        });
        userSelect?.addEventListener('change', () => {
            syncRatePlaceholder();
            syncLabor();
        });
        syncUnitFields();
        syncRatePlaceholder();
        syncLabor();
    }

    // Parts preview
    const form = document.getElementById('partForm');
    const productInput = document.getElementById('product_lookup');
    const productId = document.getElementById('product_id');
    if (!form || !productInput || !productId) return;

    const priceInput = form.querySelector('input[name="unit_price"]');
    const qtyInput = form.querySelector('input[name="qty"]');
    const discountInput = form.querySelector('input[name="discount_percent"]');
    const partPreview = document.getElementById('partAmountPreview');

    let selectedMeta = null;

    const syncPart = () => {
        if (!partPreview) return;
        const qty = parseNum(qtyInput?.value);
        const price = parseNum(priceInput?.value);
        const gross = Math.round(qty * price * 100) / 100;
        const disc = Math.min(100, Math.max(0, parseNum(discountInput?.value)));
        const amount = Math.round((gross * (1 - disc / 100)) * 100) / 100;

        const onHand = selectedMeta ? parseNum(selectedMeta.on_hand) : 0;
        const cost = selectedMeta ? parseNum(selectedMeta.cost) : 0;
        const hasProduct = String(productId.value || '').trim() !== '';
        const hasLabel = String(productInput.value || '').trim() !== '';

        const costPart = cost > 0 ? `Pašizmaksa: ${cost.toFixed(2)} EUR · ` : '';

        if (qty <= 0 || price <= 0) {
            partPreview.textContent = hasProduct ? `${costPart}Atlikums: ${onHand.toFixed(3)} · Summa: —` : 'Summa: —';
            return;
        }

        if (hasProduct) {
            partPreview.textContent = `${costPart}Atlikums: ${onHand.toFixed(3)} · Summa: ${amount.toFixed(2)} EUR${disc > 0 ? ` (-${disc.toFixed(2)}%)` : ''}`;
            return;
        }

        partPreview.textContent = hasLabel ? `Summa: ${amount.toFixed(2)} EUR${disc > 0 ? ` (-${disc.toFixed(2)}%)` : ''}` : 'Summa: —';
    };

    productInput.addEventListener('lookup:selected', (e) => {
        const meta = e.detail?.meta || {};
        selectedMeta = {
            on_hand: parseNum(meta.on_hand),
            cost: parseNum(meta.cost),
            price: parseNum(meta.price),
        };
        if (priceInput && (priceInput.value === '' || priceInput.value === '0') && selectedMeta.price > 0) {
            priceInput.value = selectedMeta.price.toFixed(2);
        }
        syncPart();
    });
    productInput.addEventListener('lookup:cleared', () => {
        selectedMeta = null;
        syncPart();
    });
    qtyInput?.addEventListener('input', syncPart);
    priceInput?.addEventListener('input', syncPart);
    discountInput?.addEventListener('input', syncPart);
    syncPart();
})();

