(() => {
    const root = document.getElementById('dashboardRoot');
    if (!root) return;

    const statsUrl = root.getAttribute('data-dashboard-stats-url') || '';
    if (!statsUrl) return;

    const nf0 = new Intl.NumberFormat(undefined, { maximumFractionDigits: 0 });
    const nf2 = new Intl.NumberFormat(undefined, { minimumFractionDigits: 2, maximumFractionDigits: 2 });

    let revenueChart = null;
    let statusChart = null;

    function cssVar(name, fallback = '') {
        const v = getComputedStyle(document.documentElement).getPropertyValue(name).trim();
        return v || fallback;
    }

    function hexToRgb(hex) {
        const s = String(hex || '').trim();
        if (!/^#?[0-9a-fA-F]{6}$/.test(s)) return null;
        const h = s.startsWith('#') ? s.slice(1) : s;
        const r = parseInt(h.slice(0, 2), 16);
        const g = parseInt(h.slice(2, 4), 16);
        const b = parseInt(h.slice(4, 6), 16);
        return Number.isFinite(r) && Number.isFinite(g) && Number.isFinite(b) ? [r, g, b] : null;
    }

    function colorFromKey(key, alpha = 0.85) {
        const k = String(key || '').trim().toLowerCase() || 'secondary';
        const rgb = cssVar(`--bs-${k}-rgb`, '');
        if (rgb) return `rgba(${rgb},${alpha})`;
        const hex = cssVar(`--bs-${k}`, '');
        const parsed = hexToRgb(hex);
        if (parsed) return `rgba(${parsed[0]},${parsed[1]},${parsed[2]},${alpha})`;
        // final fallback
        const fallbackRgb = cssVar('--bs-primary-rgb', '79,70,229');
        return `rgba(${fallbackRgb},${alpha})`;
    }

    function renderCharts(payload) {
        if (!window.Chart) return;

        const muted = cssVar('--serviss-muted', '#6b7280');

        const revEl = document.getElementById('chartRevenue');
        const stEl = document.getElementById('chartWorkOrderStatus');
        const rev = payload?.charts?.revenue14d || { labels: [], values: [] };
        const st = payload?.charts?.workOrderStatus || { labels: [], values: [] };

        if (revEl) {
            revenueChart?.destroy();
            revenueChart = new Chart(revEl, {
                type: 'line',
                data: {
                    labels: rev.labels,
                    datasets: [
                        {
                            label: 'EUR',
                            data: rev.values,
                            borderColor: colorFromKey('primary', 1),
                            backgroundColor: colorFromKey('primary', 1),
                            tension: 0.25,
                            fill: false,
                            pointRadius: 2,
                        },
                    ],
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { display: false },
                        tooltip: {
                            callbacks: {
                                label: (ctx) => ` ${nf2.format(ctx.parsed.y || 0)} EUR`,
                            },
                        },
                    },
                    scales: {
                        x: {
                            ticks: { color: muted, maxRotation: 0, autoSkip: true },
                            grid: { display: false },
                        },
                        y: {
                            ticks: { color: muted },
                            grid: { color: 'rgba(0,0,0,0.06)' },
                        },
                    },
                },
            });
        } else if (revenueChart) {
            revenueChart.destroy();
            revenueChart = null;
        }

        if (stEl) {
            statusChart?.destroy();
            const colorKeys = Array.isArray(st.colors) && st.colors.length ? st.colors : [];
            const colors = (colorKeys.length ? colorKeys : (st.codes || []).map((c) => {
                const code = String(c || '').toLowerCase();
                if (code === 'open') return 'primary';
                if (code === 'closed') return 'secondary';
                return 'warning';
            })).map((k) => colorFromKey(k, 0.85));

            statusChart = new Chart(stEl, {
                type: 'doughnut',
                data: {
                    labels: st.labels,
                    datasets: [
                        {
                            data: st.values,
                            backgroundColor: colors.length ? colors : [colorFromKey('primary', 0.85)],
                            borderWidth: 0,
                        },
                    ],
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: { position: 'bottom', labels: { color: muted, boxWidth: 10 } },
                    },
                },
            });
        } else if (statusChart) {
            statusChart.destroy();
            statusChart = null;
        }
    }

    async function load() {
        try {
            const res = await fetch(statsUrl, {
                headers: { 'X-Requested-With': 'fetch' },
                credentials: 'same-origin',
            });
            if (!res.ok) return;
            const json = await res.json();
            window.__servissDashboardPayload = json;

            const s = json?.stats || {};
            const setText = (id, value) => {
                const el = document.getElementById(id);
                if (!el) return;
                el.textContent = String(value);
            };

            setText('stat-openWorkOrders', nf0.format(s.openWorkOrders || 0));
            setText('stat-closedWithoutMechanic', nf0.format(s.closedWithoutMechanic || 0));
            setText('stat-closedWithoutInvoice', nf0.format(s.closedWithoutInvoice || 0));
            setText('stat-unpaidInvoicesCount', nf0.format(s.unpaidInvoicesCount || 0));
            setText('stat-unpaidInvoicesDue', nf2.format(s.unpaidInvoicesDue || 0));
            setText('stat-productsCount', nf0.format(s.productsCount || 0));
            setText('stat-outOfStock', nf0.format(s.outOfStock || 0));
            setText('stat-activeTireStorage', nf0.format(s.activeTireStorage || 0));
            setText('stat-unpaidDeliveriesCount', nf0.format(s.unpaidDeliveriesCount || 0));
            setText('stat-unpaidDeliveriesTotal', nf2.format(s.unpaidDeliveriesTotal || 0));

            renderCharts(json);
        } catch {
            // ignore
        }
    }

    load();

    document.addEventListener('serviss:theme', () => {
        const last = window.__servissDashboardPayload;
        if (last) {
            renderCharts(last);
            return;
        }
        load();
    });
})();

