(() => {
    const S = (window.Serviss = window.Serviss || {});
    const root = document.getElementById('calendarRoot');
    const form = document.getElementById('calendarFilterForm');
    const csrfHolder = document.getElementById('calendarCsrf');
    if (!root || !form) {
        S.initCalendarForms?.(document);
        return;
    }

    let currentRequest = null;

    const buildUrlFromForm = () => {
        const url = new URL(window.location.href);
        const view = form.querySelector('#view');
        const date = form.querySelector('#date');
        const mechanic = form.querySelector('#mechanic_id');
        const mine = form.querySelector('#mine');

        if (view && view.value) {
            url.searchParams.set('view', view.value);
        } else {
            url.searchParams.delete('view');
        }
        if (date && date.value) {
            url.searchParams.set('date', date.value);
        } else {
            url.searchParams.delete('date');
        }
        if (mechanic && mechanic.value) {
            url.searchParams.set('mechanic_id', mechanic.value);
        } else {
            url.searchParams.delete('mechanic_id');
        }
        if (mine && mine.checked) {
            url.searchParams.set('mine', '1');
        } else {
            url.searchParams.delete('mine');
        }

        return url.toString();
    };

    const syncFormFromUrl = (url) => {
        try {
            const u = new URL(url, window.location.origin);
            const params = u.searchParams;
            const view = form.querySelector('#view');
            const date = form.querySelector('#date');
            const mechanic = form.querySelector('#mechanic_id');
            const mine = form.querySelector('#mine');

            if (view) view.value = params.get('view') || 'month';
            if (date) date.value = params.get('date') || '';
            if (mechanic) mechanic.value = params.get('mechanic_id') || '';
            if (mine) mine.checked = params.get('mine') === '1';
        } catch {
            // ignore
        }
    };

    const loadCalendar = async (url, push = true) => {
        if (currentRequest) {
            currentRequest.abort();
        }
        const controller = new AbortController();
        currentRequest = controller;

        try {
            const res = await fetch(url, {
                headers: { 'X-Requested-With': 'fetch' },
                credentials: 'same-origin',
                signal: controller.signal,
            });
            if (!res.ok) {
                window.location.href = url;
                return;
            }
            const json = await res.json();
            if (json?.html) {
                root.innerHTML = json.html;
                if (push) {
                    window.history.pushState({ calendarUrl: url }, '', url);
                }
                syncFormFromUrl(url);
                enableDragAndDrop();
                S.initCalendarForms?.(document);
            } else {
                window.location.href = url;
            }
        } catch (e) {
            if (e?.name !== 'AbortError') {
                window.location.href = url;
            }
        } finally {
            if (currentRequest === controller) {
                currentRequest = null;
            }
        }
    };

    const submit = (push = true) => {
        const url = buildUrlFromForm();
        loadCalendar(url, push);
    };

    form.addEventListener('change', (e) => {
        if (!(e.target instanceof HTMLSelectElement || e.target instanceof HTMLInputElement)) return;
        submit(true);
    });

    form.addEventListener('submit', (e) => {
        e.preventDefault();
        submit(true);
    });

    document.addEventListener('click', (e) => {
        const link = e.target.closest('[data-calendar-nav="1"]');
        if (!link) return;
        const url = link.getAttribute('href');
        if (!url) return;
        e.preventDefault();
        loadCalendar(url, true);
    });

    window.addEventListener('popstate', (e) => {
        const url = e.state?.calendarUrl || window.location.href;
        loadCalendar(url, false);
    });

    S.calendarRefresh = () => {
        submit(false);
    };

    S.initCalendarForms?.(document);

    const getCsrf = () => {
        const input = csrfHolder?.querySelector('input');
        if (!(input instanceof HTMLInputElement)) return { name: '', value: '' };
        return { name: input.name, value: input.value };
    };

    const setCsrf = (name, value) => {
        const input = csrfHolder?.querySelector('input');
        if (!(input instanceof HTMLInputElement)) return;
        if (name && input.name !== name) {
            input.name = name;
        }
        if (value) {
            input.value = value;
        }
    };

    const formatDateTime = (dateObj) => {
        const pad = (n) => String(n).padStart(2, '0');
        const y = dateObj.getFullYear();
        const m = pad(dateObj.getMonth() + 1);
        const d = pad(dateObj.getDate());
        const hh = pad(dateObj.getHours());
        const mm = pad(dateObj.getMinutes());
        const ss = pad(dateObj.getSeconds());
        return `${y}-${m}-${d} ${hh}:${mm}:${ss}`;
    };

    const parseDateTime = (value) => {
        const v = String(value || '').trim();
        if (!v) return null;
        const norm = v.includes('T') ? v : v.replace(' ', 'T');
        const dt = new Date(norm);
        return Number.isNaN(dt.getTime()) ? null : dt;
    };

    const enableDragAndDrop = () => {
        const events = root.querySelectorAll('[data-event-id]');
        events.forEach((el) => {
            const canDrag = el.getAttribute('data-can-drag') === '1';
            if (!canDrag) return;
            if (el.dataset.dragReady === '1') return;
            el.dataset.dragReady = '1';

            el.addEventListener('dragstart', (e) => {
                const id = el.getAttribute('data-event-id') || '';
                const startAt = el.getAttribute('data-start-at') || '';
                const endAt = el.getAttribute('data-end-at') || '';
                const payload = JSON.stringify({ id, startAt, endAt });
                e.dataTransfer?.setData('text/plain', payload);
                el.classList.add('calendar-dragging');
            });
            el.addEventListener('dragend', () => {
                el.classList.remove('calendar-dragging');
            });
        });

        root.querySelectorAll('[data-calendar-drop="1"]').forEach((cell) => {
            if (cell.dataset.dropReady === '1') return;
            cell.dataset.dropReady = '1';

            cell.addEventListener('dragover', (e) => {
                e.preventDefault();
                cell.classList.add('calendar-drop-hover');
            });
            cell.addEventListener('dragleave', () => {
                cell.classList.remove('calendar-drop-hover');
            });
            cell.addEventListener('drop', async (e) => {
                e.preventDefault();
                cell.classList.remove('calendar-drop-hover');
                const raw = e.dataTransfer?.getData('text/plain') || '';
                if (!raw) return;
                let data = null;
                try {
                    data = JSON.parse(raw);
                } catch {
                    data = null;
                }
                if (!data?.id) return;

                const dropDate = cell.getAttribute('data-date');
                if (!dropDate) return;

                const start = parseDateTime(data.startAt);
                const end = parseDateTime(data.endAt);
                const durationMs = start && end ? Math.max(0, end.getTime() - start.getTime()) : 60 * 60 * 1000;
                const time = start ? `${String(start.getHours()).padStart(2, '0')}:${String(start.getMinutes()).padStart(2, '0')}:${String(start.getSeconds()).padStart(2, '0')}` : '09:00:00';
                const newStart = new Date(`${dropDate}T${time}`);
                const newEnd = new Date(newStart.getTime() + durationMs);

                if (Number.isNaN(newStart.getTime()) || Number.isNaN(newEnd.getTime())) return;

                const formData = new FormData();
                formData.append('start_at', formatDateTime(newStart));
                formData.append('end_at', formatDateTime(newEnd));
                const csrf = getCsrf();
                if (csrf.name && csrf.value) {
                    formData.append(csrf.name, csrf.value);
                }

                try {
                    const res = await fetch(`/calendar/${data.id}/move`, {
                        method: 'POST',
                        body: formData,
                        headers: { 'X-Requested-With': 'fetch' },
                        credentials: 'same-origin',
                    });
                    const json = await res.json().catch(() => null);
                    if (json?.csrfToken && json?.csrfHash) {
                        setCsrf(json.csrfToken, json.csrfHash);
                        S.updateCsrf?.(json.csrfToken, json.csrfHash);
                    }
                    if (!res.ok) {
                        return;
                    }
                    if (json?.calendarRefresh) {
                        S.calendarRefresh?.();
                    }
                } catch {
                    // ignore
                }
            });
        });
    };

    enableDragAndDrop();
})();
