(() => {
    window.Serviss = window.Serviss || {};

    const S = window.Serviss;

    S.cssEscape = (value) => {
        if (window.CSS && typeof CSS.escape === 'function') {
            return CSS.escape(String(value));
        }
        // Minimal fallback
        return String(value).replace(/["\\]/g, '\\$&');
    };

    S.clearFormErrors = (form) => {
        if (!form) return;
        form.querySelectorAll('.is-invalid').forEach((el) => el.classList.remove('is-invalid'));
        form.querySelectorAll('[data-ajax-error="1"]').forEach((el) => el.remove());
    };

    S.showFormErrors = (form, errors) => {
        if (!form) return;
        S.clearFormErrors(form);
        Object.entries(errors || {}).forEach(([name, msg]) => {
            const selector = `[name="${S.cssEscape(name)}"]`;
            const aliasSelector = `[data-error-alias="${S.cssEscape(name)}"]`;
            const field = form.querySelector(aliasSelector) || form.querySelector(selector);
            if (!field) return;

            // If the field is inside a collapsed <details>, open it so the user can see the error.
            const details = field.closest?.('details');
            if (details) details.open = true;

            field.classList.add('is-invalid');
            const fb = document.createElement('div');
            fb.className = 'invalid-feedback';
            fb.setAttribute('data-ajax-error', '1');
            fb.textContent = String(msg || 'Nepareiza vērtība.');
            const group = field.closest?.('.input-group') || null;
            (group || field).insertAdjacentElement('afterend', fb);
        });
    };

    S.updateCsrf = (tokenName, tokenValue) => {
        if (!tokenName || !tokenValue) return;
        const selector = `input[name="${S.cssEscape(tokenName)}"]`;
        document.querySelectorAll(selector).forEach((el) => {
            el.value = tokenValue;
        });
    };

    S.flash = (container, message, type = 'success') => {
        if (!container || !message) return;
        const el = document.createElement('div');
        el.className = `alert alert-${type} py-2`;
        el.textContent = String(message);
        container.prepend(el);
        window.setTimeout(() => el.remove(), 3500);
    };

    S.initLookups = (root = document) => {
        const debounce = (fn, wait = 250) => {
            let t = null;
            return (...args) => {
                if (t) window.clearTimeout(t);
                t = window.setTimeout(() => fn(...args), wait);
            };
        };

        const inputs = root.querySelectorAll('input[data-lookup]');
        inputs.forEach((input) => {
            if (!(input instanceof HTMLInputElement)) return;
            if (input.dataset.lookupReady === '1') return;
            input.dataset.lookupReady = '1';

            const endpoint = String(input.getAttribute('data-lookup') || '').trim();
            if (!endpoint) return;

            const listId = input.getAttribute('list');
            const datalist = listId ? document.getElementById(listId) : null;
            const scopeSelector = input.getAttribute('data-lookup-scope');
            const scope = scopeSelector ? input.closest(scopeSelector) : null;
            const targetSelector = input.getAttribute('data-lookup-target');
            const target = targetSelector
                ? (scope?.querySelector(targetSelector) || root.querySelector(targetSelector) || document.querySelector(targetSelector))
                : null;

            const minLen = parseInt(String(input.getAttribute('data-lookup-min') || '2'), 10) || 2;
            const limit = parseInt(String(input.getAttribute('data-lookup-limit') || '20'), 10) || 20;
            let map = new Map();
            let dirty = false;

            const dispatch = (name, detail) => {
                try {
                    input.dispatchEvent(new CustomEvent(name, { detail, bubbles: true }));
                } catch {
                    // ignore
                }
            };

            const applySelection = (force = false) => {
                if (!force && !dirty) return;
                const value = String(input.value || '').trim();
                const row = map.get(value);
                if (target) {
                    target.value = row ? String(row.id || '') : '';
                }
                if (row) {
                    const costTarget = input.getAttribute('data-lookup-cost-target');
                    if (costTarget && row.meta && row.meta.cost !== null && row.meta.cost !== undefined) {
                        const scopeRoot = scope || input.closest('form') || document;
                        const costEl = scopeRoot.querySelector(costTarget);
                        if (costEl && (costEl.value === '' || costEl.value === '0')) {
                            costEl.value = String(row.meta.cost);
                        }
                    }
                    dispatch('lookup:selected', row);
                } else {
                    dispatch('lookup:cleared', {});
                }
            };

            const render = (rows) => {
                map = new Map();
                if (datalist) datalist.innerHTML = '';
                (rows || []).forEach((row) => {
                    const label = String(row?.label || '').trim();
                    if (!label) return;
                    map.set(label, row);
                    if (datalist) {
                        const opt = document.createElement('option');
                        opt.value = label;
                        datalist.appendChild(opt);
                    }
                });
            };

            const load = async () => {
                const q = String(input.value || '').trim();
                if (q.length < minLen) {
                    render([]);
                    return;
                }

                const url = new URL(`/lookup/${endpoint}`, window.location.origin);
                url.searchParams.set('q', q);
                url.searchParams.set('limit', String(limit));

                try {
                    const res = await fetch(url.toString(), {
                        credentials: 'same-origin',
                        headers: { 'X-Requested-With': 'fetch' },
                    });
                    if (!res.ok) return;
                    const json = await res.json();
                    render(json?.rows || []);
                    applySelection(true);
                } catch {
                    // ignore
                }
            };

            const debounced = debounce(load, 250);

            input.addEventListener('input', () => {
                dirty = true;
                debounced();
            });
            input.addEventListener('change', () => applySelection());
            input.addEventListener('blur', () => applySelection());
        });
    };

    S.initCalendarForms = (root = document) => {
        root.querySelectorAll('[data-calendar-form="1"]').forEach((form) => {
            if (!(form instanceof HTMLFormElement)) return;

            const toggle = form.querySelector('[data-calendar-type-toggle="1"]');
            const sections = Array.from(form.querySelectorAll('[data-calendar-type]'));
            if (!toggle || sections.length === 0) return;

            const getType = () => String(toggle.value || '').trim() || 'service';

            const apply = () => {
                const t = getType();
                sections.forEach((el) => {
                    const attr = String(el.getAttribute('data-calendar-type') || '');
                    const types = attr.split(',').map((v) => v.trim()).filter(Boolean);
                    el.style.display = types.includes(t) ? '' : 'none';
                });

                const privacy = form.querySelector('#is_private');
                if (privacy instanceof HTMLInputElement && t === 'personal' && !privacy.checked) {
                    privacy.checked = true;
                }
            };

            toggle.addEventListener('change', apply);
            apply();
        });
    };

    // Client <-> Vehicle helper: auto-select most recent vehicle for a client, auto-select owner for a vehicle,
    // and show a mismatch warning (allowed).
    S.initClientVehiclePairs = (root = document) => {
        root.querySelectorAll('[data-client-vehicle-pair="1"]').forEach((scope) => {
            const clientIdEl = scope.querySelector('[data-pair-client-id="1"]');
            const vehicleIdEl = scope.querySelector('[data-pair-vehicle-id="1"]');
            const clientLookup = scope.querySelector('[data-pair-client-lookup="1"]');
            const vehicleLookup = scope.querySelector('[data-pair-vehicle-lookup="1"]');
            if (!(clientIdEl instanceof HTMLInputElement) || !(vehicleIdEl instanceof HTMLInputElement)) return;
            if (!(clientLookup instanceof HTMLInputElement) || !(vehicleLookup instanceof HTMLInputElement)) return;

            if (scope.dataset.pairReady === '1') return;
            scope.dataset.pairReady = '1';

            const mismatch = scope.querySelector('[data-pair-mismatch="1"]');
            const clientNameEl = scope.querySelector('[data-pair-client-name="1"]');
            const ownerNameEl = scope.querySelector('[data-pair-owner-name="1"]');

            const clientVehiclesUrl = String(scope.getAttribute('data-client-vehicles-url') || '').trim();

            let clientTouched = false;
            let vehicleTouched = false;
            let vehicleOwnerId = parseInt(String(scope.getAttribute('data-initial-vehicle-client-id') || '0'), 10) || 0;
            let vehicleOwnerName = '';

            const readId = (el) => {
                const v = String(el.value || '').trim();
                const n = parseInt(v, 10);
                return Number.isFinite(n) && n > 0 ? n : 0;
            };

            const syncMismatch = () => {
                if (!(mismatch instanceof HTMLElement)) return;
                const cid = readId(clientIdEl);
                const vid = readId(vehicleIdEl);
                const ownerId = vehicleOwnerId || 0;
                const show = cid > 0 && vid > 0 && ownerId > 0 && cid !== ownerId;
                mismatch.classList.toggle('d-none', !show);
                if (show) {
                    if (clientNameEl) clientNameEl.textContent = String(clientLookup.value || `ID ${cid}`);
                    if (ownerNameEl) ownerNameEl.textContent = vehicleOwnerName || `ID ${ownerId}`;
                }
            };

            const setVehicle = (row) => {
                if (!row) return;
                const id = parseInt(String(row.id || '0'), 10) || 0;
                if (id <= 0) return;
                vehicleIdEl.value = String(id);
                vehicleLookup.value = String(row.label || '');
                const meta = row.meta || {};
                vehicleOwnerId = parseInt(String(meta.client_id || '0'), 10) || 0;
                vehicleOwnerName = String(meta.client_name || '');
                syncMismatch();
            };

            const autoPickLatestVehicle = async (clientId) => {
                if (!clientVehiclesUrl) return;
                if (clientId <= 0) return;
                try {
                    const url = `${clientVehiclesUrl}/${clientId}?limit=1`;
                    const res = await fetch(url, {
                        headers: { 'X-Requested-With': 'fetch' },
                        credentials: 'same-origin',
                    });
                    if (!res.ok) return;
                    const json = await res.json();
                    const row = (json?.rows || [])[0] || null;
                    if (!row) return;
                    setVehicle(row);
                } catch {
                    // ignore
                }
            };

            clientLookup.addEventListener('lookup:selected', (e) => {
                clientTouched = true;
                const row = e.detail || {};
                const cid = parseInt(String(row.id || clientIdEl.value || '0'), 10) || 0;
                if (cid > 0) clientIdEl.value = String(cid);

                const currentVehicleId = readId(vehicleIdEl);
                if (currentVehicleId <= 0 || !vehicleTouched) {
                    autoPickLatestVehicle(cid);
                }
                syncMismatch();
            });
            clientLookup.addEventListener('lookup:cleared', () => {
                clientTouched = true;
                syncMismatch();
            });

            vehicleLookup.addEventListener('lookup:selected', (e) => {
                vehicleTouched = true;
                const row = e.detail || {};
                const meta = row.meta || {};
                vehicleOwnerId = parseInt(String(meta.client_id || '0'), 10) || 0;
                vehicleOwnerName = String(meta?.client?.name || '');

                const currentClientId = readId(clientIdEl);
                if (currentClientId <= 0 || !clientTouched) {
                    const client = meta.client || null;
                    const cid = parseInt(String(meta.client_id || client?.id || '0'), 10) || 0;
                    if (cid > 0) clientIdEl.value = String(cid);
                    if (client && client.name) {
                        clientLookup.value = String(client.name);
                    }
                }
                syncMismatch();
            });
            vehicleLookup.addEventListener('lookup:cleared', () => {
                vehicleTouched = true;
                vehicleOwnerId = 0;
                vehicleOwnerName = '';
                syncMismatch();
            });

            // Buttons to open "new client/vehicle" sections (best-effort: open details + clear selection)
            scope.querySelectorAll('[data-open-new-client="1"]').forEach((btn) => {
                btn.addEventListener('click', () => {
                    const details = document.getElementById('quoteNewClientDetails');
                    if (details instanceof HTMLDetailsElement) details.open = true;
                });
            });
            scope.querySelectorAll('[data-open-new-vehicle="1"]').forEach((btn) => {
                btn.addEventListener('click', () => {
                    const details = document.getElementById('quoteNewVehicleDetails');
                    if (details instanceof HTMLDetailsElement) details.open = true;
                    // Clear selection so server-side can create the new vehicle without ambiguity.
                    vehicleIdEl.value = '';
                    vehicleLookup.value = '';
                    vehicleOwnerId = 0;
                    vehicleOwnerName = '';
                    syncMismatch();
                });
            });

            // Initial paint (mismatch only works if server provided vehicle owner id)
            syncMismatch();
        });
    };

    S.applyTheme = (applyTheme) => {
        if (!applyTheme) return;
        const light = applyTheme.light_primary || '#4f46e5';
        const dark = applyTheme.dark_primary || '#818cf8';
        const fontKey = applyTheme.font || 'system';

        const clamp = (n) => Math.max(0, Math.min(255, n));
        const toRgb = (hex) => {
            const h = String(hex || '').trim().replace('#', '');
            if (!/^[0-9a-fA-F]{6}$/.test(h)) return { r: 79, g: 70, b: 229 };
            return {
                r: parseInt(h.slice(0, 2), 16),
                g: parseInt(h.slice(2, 4), 16),
                b: parseInt(h.slice(4, 6), 16),
            };
        };
        const toHex = ({ r, g, b }) =>
            `#${clamp(r).toString(16).padStart(2, '0')}${clamp(g).toString(16).padStart(2, '0')}${clamp(b)
                .toString(16)
                .padStart(2, '0')}`;
        const mix = (hex1, hex2, weight2) => {
            const a = toRgb(hex1);
            const b = toRgb(hex2);
            const w1 = 1 - weight2;
            return toHex({
                r: Math.round(a.r * w1 + b.r * weight2),
                g: Math.round(a.g * w1 + b.g * weight2),
                b: Math.round(a.b * w1 + b.b * weight2),
            });
        };

        const rgb = (hex) => {
            const c = toRgb(hex);
            return `${c.r}, ${c.g}, ${c.b}`;
        };

        const fontMap = {
            system: 'ui-sans-serif, system-ui, -apple-system, "Segoe UI", Roboto, Arial, "Apple Color Emoji", "Segoe UI Emoji"',
            segoe: '"Segoe UI", system-ui, -apple-system, Roboto, Arial, "Apple Color Emoji", "Segoe UI Emoji"',
            roboto: 'Roboto, "Segoe UI", system-ui, -apple-system, Arial, "Apple Color Emoji", "Segoe UI Emoji"',
            inter: 'Inter, "Segoe UI", system-ui, -apple-system, Roboto, Arial, "Apple Color Emoji", "Segoe UI Emoji"',
            manrope: 'Manrope, "Segoe UI", system-ui, -apple-system, Roboto, Arial, "Apple Color Emoji", "Segoe UI Emoji"',
            noto: '"Noto Sans", "Segoe UI", system-ui, -apple-system, Roboto, Arial, "Apple Color Emoji", "Segoe UI Emoji"',
            sourcesans: '"Source Sans 3", "Segoe UI", system-ui, -apple-system, Roboto, Arial, "Apple Color Emoji", "Segoe UI Emoji"',
            plusjakarta: '"Plus Jakarta Sans", "Segoe UI", system-ui, -apple-system, Roboto, Arial, "Apple Color Emoji", "Segoe UI Emoji"',
            ibmplex: '"IBM Plex Sans", "Segoe UI", system-ui, -apple-system, Roboto, Arial, "Apple Color Emoji", "Segoe UI Emoji"',
            worksans: '"Work Sans", "Segoe UI", system-ui, -apple-system, Roboto, Arial, "Apple Color Emoji", "Segoe UI Emoji"',
        };

        const googleHref = {
            roboto: 'https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;600;700&display=swap',
            inter: 'https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800&display=swap',
            manrope: 'https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700;800&display=swap',
            noto: 'https://fonts.googleapis.com/css2?family=Noto+Sans:wght@400;500;600;700&display=swap',
            sourcesans: 'https://fonts.googleapis.com/css2?family=Source+Sans+3:wght@400;500;600;700;800&display=swap',
            plusjakarta: 'https://fonts.googleapis.com/css2?family=Plus+Jakarta+Sans:wght@400;500;600;700;800&display=swap',
            ibmplex: 'https://fonts.googleapis.com/css2?family=IBM+Plex+Sans:wght@400;500;600;700&display=swap',
            worksans: 'https://fonts.googleapis.com/css2?family=Work+Sans:wght@400;500;600;700;800&display=swap',
        };

        const href = googleHref[fontKey] || '';
        if (href) {
            let link = document.getElementById('servissGoogleFont');
            if (!link) {
                link = document.createElement('link');
                link.id = 'servissGoogleFont';
                link.rel = 'stylesheet';
                document.head.appendChild(link);
            }
            link.href = href;
        }

        let style = document.getElementById('servissThemeVars');
        if (!style) {
            style = document.createElement('style');
            style.id = 'servissThemeVars';
            document.head.appendChild(style);
        }

        const fontFamily = fontMap[fontKey] || fontMap.system;
        const lightTextEmphasis = mix(light, '#000000', 0.35);
        const lightBgSubtle = mix(light, '#ffffff', 0.92);
        const lightBorderSubtle = mix(light, '#ffffff', 0.80);
        const lightLinkHover = mix(light, '#000000', 0.15);

        const darkTextEmphasis = mix(dark, '#ffffff', 0.20);
        const darkBgSubtle = mix(dark, '#000000', 0.82);
        const darkBorderSubtle = mix(dark, '#000000', 0.60);
        const darkLinkHover = mix(dark, '#ffffff', 0.18);

        const lightBgKey = String(applyTheme.light_bg || 'soft')
            .trim()
            .toLowerCase();
        const darkBgKey = String(applyTheme.dark_bg || 'deep')
            .trim()
            .toLowerCase();
        const patternKey = String(applyTheme.bg_pattern || 'none')
            .trim()
            .toLowerCase();
        const topbarPos = applyTheme.topbar_sticky ? 'sticky' : 'static';

        const lightBgMap = {
            soft: { body: '#f5f6f8', border: '#e7e9ee', secondary: '#ffffff', tertiary: '#ffffff', surface: '#ffffff', surface2: '#ffffff' },
            bright: { body: '#ffffff', border: '#e7e9ee', secondary: '#ffffff', tertiary: '#ffffff', surface: '#ffffff', surface2: '#ffffff' },
            paper: { body: '#f7f4ee', border: '#e6e0d6', secondary: '#ffffff', tertiary: '#ffffff', surface: '#ffffff', surface2: '#ffffff' },
        };
        const darkBgMap = {
            deep: { body: '#0b0f1a', border: '#22283a', secondary: '#111625', tertiary: '#151a2a', surface: '#151a2a', surface2: '#0f1424' },
            mid: { body: '#0f1424', border: '#2a3248', secondary: '#141b2d', tertiary: '#1b2337', surface: '#1b2337', surface2: '#141b2d' },
            soft: { body: '#151a2a', border: '#303a53', secondary: '#1b2337', tertiary: '#222b41', surface: '#222b41', surface2: '#1b2337' },
        };
        const lb = lightBgMap[lightBgKey] || lightBgMap.soft;
        const db = darkBgMap[darkBgKey] || darkBgMap.deep;

        const patterns = {
            none: { pattern: 'none', size: 'auto', position: '0 0', attachment: 'scroll' },
            dots: {
                pattern: 'radial-gradient(rgba(var(--bs-primary-rgb), 0.10) 1px, transparent 1px)',
                size: '18px 18px',
                position: '0 0',
                attachment: 'scroll',
            },
            gradient: {
                pattern:
                    'radial-gradient(1200px 600px at 10% 10%, rgba(var(--bs-primary-rgb), 0.12), transparent 55%), radial-gradient(900px 520px at 90% 20%, rgba(16, 185, 129, 0.10), transparent 55%)',
                size: 'auto',
                position: '0 0',
                attachment: 'scroll',
            },
        };
        const pat = patterns[patternKey] || patterns.none;

        style.textContent =
            `:root{--serviss-font:${fontFamily};` +
            `--serviss-bg-pattern:${pat.pattern};--serviss-bg-size:${pat.size};--serviss-bg-position:${pat.position};--serviss-bg-attachment:${pat.attachment};` +
            `--serviss-topbar-position:${topbarPos};}` +
            `:root,[data-bs-theme="light"]{` +
            `--bs-body-bg:${lb.body};--bs-border-color:${lb.border};--bs-secondary-bg:${lb.secondary};--bs-tertiary-bg:${lb.tertiary};` +
            `--serviss-surface:${lb.surface};--serviss-surface-2:${lb.surface2};` +
            `--bs-primary:${light};--bs-primary-rgb:${rgb(light)};` +
            `--bs-primary-text-emphasis:${lightTextEmphasis};` +
            `--bs-primary-bg-subtle:${lightBgSubtle};` +
            `--bs-primary-border-subtle:${lightBorderSubtle};` +
            `--bs-link-color:${light};--bs-link-color-rgb:${rgb(light)};` +
            `--bs-link-hover-color:${lightLinkHover};--bs-link-hover-color-rgb:${rgb(lightLinkHover)};` +
            `}` +
            `[data-bs-theme="dark"]{` +
            `--bs-body-bg:${db.body};--bs-border-color:${db.border};--bs-secondary-bg:${db.secondary};--bs-tertiary-bg:${db.tertiary};` +
            `--serviss-surface:${db.surface};--serviss-surface-2:${db.surface2};` +
            `--bs-primary:${dark};--bs-primary-rgb:${rgb(dark)};` +
            `--bs-primary-text-emphasis:${darkTextEmphasis};` +
            `--bs-primary-bg-subtle:${darkBgSubtle};` +
            `--bs-primary-border-subtle:${darkBorderSubtle};` +
            `--bs-link-color:${dark};--bs-link-color-rgb:${rgb(dark)};` +
            `--bs-link-hover-color:${darkLinkHover};--bs-link-hover-color-rgb:${rgb(darkLinkHover)};` +
            `}`;

        try {
            const current = document.documentElement.getAttribute('data-bs-theme') || 'light';
            document.dispatchEvent(new CustomEvent('serviss:theme', { detail: { theme: current } }));
        } catch {
            // ignore
        }
    };

    S.updateBrand = (brandName, brandLogoUrl) => {
        const nameEl = document.getElementById('appBrandName');
        if (nameEl && brandName) nameEl.textContent = brandName;

        const img = document.getElementById('appBrandLogo');
        const initial = document.getElementById('appBrandInitial');

        if (!img) return;
        if (brandLogoUrl) {
            img.src = brandLogoUrl;
            img.classList.remove('d-none');
            if (initial) initial.classList.add('d-none');
            return;
        }
        img.classList.add('d-none');
        if (initial) initial.classList.remove('d-none');
    };
})();

